/*******************************************************************************
 * Copyright (c) 2000, 2013 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.ui.tests.refactoring;

import java.util.ArrayList;
import java.util.List;

import org.junit.Assert;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;

import org.eclipse.core.resources.IResource;

import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.ltk.core.refactoring.participants.CheckConditionsContext;
import org.eclipse.ltk.core.refactoring.participants.CopyArguments;
import org.eclipse.ltk.core.refactoring.participants.CopyParticipant;
import org.eclipse.ltk.core.refactoring.participants.ISharableParticipant;
import org.eclipse.ltk.core.refactoring.participants.RefactoringArguments;
import org.eclipse.ltk.core.refactoring.participants.ReorgExecutionLog;

import org.eclipse.jdt.core.IJavaElement;

import org.eclipse.jdt.internal.corext.util.JavaElementResourceMapping;

public class TestCopyParticipantShared extends CopyParticipant implements ISharableParticipant {

	static TestCopyParticipantShared fgInstance;

	List<Object> fElements= new ArrayList<>(3);
	List<String> fHandles= new ArrayList<>(3);
	List<RefactoringArguments> fArguments= new ArrayList<>(3);

	@Override
	public boolean initialize(Object element) {
		fgInstance= this;
		fElements.add(element);
		fArguments.add(getArguments());
		if (element instanceof IJavaElement) {
			fHandles.add(((IJavaElement)element).getHandleIdentifier());
		} else if (element instanceof IResource) {
			fHandles.add(((IResource)element).getFullPath().toString());
		} else if (element instanceof JavaElementResourceMapping) {
			fHandles.add(((JavaElementResourceMapping)element).
				getJavaElement().getHandleIdentifier() + "_mapping");
		}
		return true;
	}

	@Override
	public void addElement(Object element, RefactoringArguments args) {
		fElements.add(element);
		fArguments.add(args);
		if (element instanceof IJavaElement) {
			fHandles.add(((IJavaElement)element).getHandleIdentifier());
		} else if (element instanceof IResource) {
			fHandles.add(((IResource)element).getFullPath().toString());
		} else if (element instanceof JavaElementResourceMapping) {
			fHandles.add(((JavaElementResourceMapping)element).getJavaElement().getHandleIdentifier() + "_mapping");
		}
	}

	@Override
	public String getName() {
		return getClass().getName();
	}

	@Override
	public RefactoringStatus checkConditions(IProgressMonitor pm, CheckConditionsContext context) {
		return new RefactoringStatus();
	}

	@Override
	public Change createChange(IProgressMonitor pm) throws CoreException {
		return null;
	}

	public static void testNumberOfElements(int expected) {
		if (expected == 0) {
			Assert.assertNull(fgInstance);
		} else {
			Assert.assertEquals(expected, fgInstance.fElements.size());
			Assert.assertEquals(expected, fgInstance.fArguments.size());
		}
	}

	public static void testArguments(CopyArguments[] args) {
		testNumberOfElements(args.length);
		for (int i= 0; i < args.length; i++) {
			CopyArguments expected= args[i];
			CopyArguments actual= (CopyArguments)fgInstance.fArguments.get(i);
			compareArguments(expected, actual);
		}
	}

	public static void compareArguments(CopyArguments expected, CopyArguments actual) {
		Assert.assertEquals("Destination: ", expected.getDestination(), actual.getDestination());
		compareExecutionLog(expected.getExecutionLog(), actual.getExecutionLog());
	}

	private static void compareExecutionLog(ReorgExecutionLog expected, ReorgExecutionLog actual) {
		Assert.assertEquals("Canceled: ", expected.isCanceled(), actual.isCanceled());
		Object[] expectedRenamed= expected.getRenamedElements();
		Object[] actualRenamed= actual.getRenamedElements();
		Assert.assertEquals(expectedRenamed.length, actualRenamed.length);
		for (int j= 0; j < expectedRenamed.length; j++) {
			Assert.assertEquals(expected.getNewName(expectedRenamed[j]), actual.getNewName(actualRenamed[j]));
		}
		Object[] expectedProcessed= expected.getProcessedElements();
		Object[] actualProcessed= actual.getProcessedElements();
		Assert.assertEquals(expectedProcessed.length, actualProcessed.length);
		for (int j= 0; j < expectedProcessed.length; j++) {
			Assert.assertEquals(expectedProcessed[j], actualProcessed[j]);
		}
	}

	public static void reset() {
		fgInstance= null;
	}
}
